using System;
using System.Collections.Generic;
using System.Text;

namespace FS28BluetoothSlaveModeDemo
{
    public delegate void AddCommandListHandle( byte nFlag, byte[] Command );
    public delegate void ShowCommandListHandle();
    public delegate void ShowTextMessageHandle(string strMessage);

    public class FamComm
    {
        public event AddCommandListHandle OnAddCommandList;
        public event ShowCommandListHandle OnShowCommandList;
        public event ShowTextMessageHandle OnShowTextMessage;

        private FamSerialComm m_commSerial;
        private static byte m_nErrorCode;
        private string m_strErrorMessage;
        private byte[] m_RxCmd;
        private bool m_bMatchWithLog = false;

        public FamComm()
        {
            m_RxCmd = new byte[13];
            m_strErrorMessage = "";
            m_nErrorCode = 0;
            m_commSerial = new FamSerialComm();
        }

        public void AddCommandEventHandle( )
        {
            m_commSerial.OnAddCommandList += this.OnAddCommandList;
            m_commSerial.OnShowCommandList += this.OnShowCommandList;
        }

        public void RemoveCommandEventHandle( )
        {
            m_commSerial.OnAddCommandList -= this.OnAddCommandList;
            m_commSerial.OnShowCommandList -= this.OnShowCommandList;
        }

        public void AddTextEventHandle()
        {
            m_commSerial.OnShowTextMessage += this.OnShowTextMessage;
        }

        public string ComPort
        {
            get { return m_commSerial.ComPort; }
            set { m_commSerial.ComPort = value; }
        }

        public int Baudrate
        {
            get { return m_commSerial.Baudrate; }
            set { m_commSerial.Baudrate = value; }
        }

        public bool MatchWithLog
        {
            get { return m_bMatchWithLog; }
            set { m_bMatchWithLog = value; }
        }

        //Fam Communication Commands
        public byte PrepareConnection()
        {
            m_nErrorCode = m_commSerial.PrepareComPort();
            return m_nErrorCode;
        }

        public byte CheckConnection()
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_GET_VERSION, 0, 0, 0, null, null);
            return m_nErrorCode;
        }

        public void CloseConnection()
        {
            m_commSerial.CloseComPort();
        }

        public byte CommunicateWithFAC(byte nCommand, uint param1, uint param2, byte nFlag, byte[] TxBuf, byte[] RxBuf)
        {
            m_nErrorCode = m_commSerial.CommunicateWithFAC(nCommand, param1, param2, nFlag, m_RxCmd, TxBuf, RxBuf);
            return m_nErrorCode;
        }

        public byte FamIsFingerPresent()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_CHECK_FINGER, 0, 0, 0, null, null);
        }

        public byte FamCaptureImage(bool bPIV, ref uint nContrast, ref uint nBrightness)
        {
            uint nC = 0;
            uint nB = 0;
            uint nParam1 = 0;
            if (bPIV)
                nParam1 = 0x08;
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_CAPTURE_IMAGE, nParam1, 0, 0, null, null);
            if (m_nErrorCode == 0)
            {
                nC = (uint)(m_RxCmd[2] + (m_RxCmd[3] << 8) + (m_RxCmd[4] << 16) + (m_RxCmd[5] << 24));
                nB = (uint)(m_RxCmd[6] + (m_RxCmd[7] << 8) + (m_RxCmd[8] << 16) + (m_RxCmd[9] << 24));
            }
            nContrast = nC;
            nBrightness = nB;
            return m_nErrorCode;
        }

        public byte FamConvertWSQ(uint bitrate, ref uint WSQ_size) // (bitrate 0x4B-0xFF), to compress RAW image to WSQ image. 0x4B for 0.75(15:1), 0x96 for 1.5 (8.69:1) 
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_CONVERT_TO_WSQ, bitrate, 0, 1, null, null); //flag is set to 1 for WSQ.
            if (m_nErrorCode == 0)
            {
                WSQ_size = (uint)(m_RxCmd[6] + (m_RxCmd[7] << 8) + (m_RxCmd[8] << 16) + (m_RxCmd[9] << 24));
            }
            return m_nErrorCode;
        }

        public byte FamDownload_WSQ(byte[] pImage, uint WSQ_size)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_WSQ_IMAGE, 0, WSQ_size, 0, null, pImage); 
        }

        public byte FamDownloadRAWImage(byte[] pImage)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_RAW_IMAGE, 0, 320 * 480, 0, null, pImage);
        }

        public byte FamDownloadRAWImage_Size_Offset(byte[] pImage, uint nImgSize, uint nOffset)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_RAW_IMAGE, nOffset, nImgSize, 0, null, pImage);
        }

        public byte FamProcessImage(byte nFlag)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_PROCESS_IMAGE, 0, 0, nFlag, null, null);
        }

        public byte FamGetSampleSize(ref uint SampleSize, byte nFlag)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_SAMPLE, 0, 0, nFlag, null, null);
            if (m_nErrorCode == 0)
            {
                SampleSize = (uint)(m_RxCmd[6] + (m_RxCmd[7] << 8) + (m_RxCmd[8] << 16) + (m_RxCmd[9] << 24));
            }
            return m_nErrorCode;
        }

        public byte FamDownload_Sample(byte[] pSample,uint sampleSize, byte nFlag)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_SAMPLE, 0, sampleSize, nFlag, null, pSample);
        }

        public byte FamStoreSample(byte nSample)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_STORE_SAMPLE, nSample, 0, 0, null, null);
        }

        public byte FamStoreTemplate(uint nID_L, uint nID_H, byte nUType)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_STORE_TEMPLATE, nID_L, nID_H, nUType, null, null);
        }

        public byte FamVerify(uint nID_L, uint nID_H)
        {
            byte nFlag = FamDefs.FLAG_1_1_MATCH;
            if (m_bMatchWithLog)
                nFlag = FamDefs.FLAG_1_1_MATCH_LOG;
            return CommunicateWithFAC(FamDefs.COMMAND_MATCH_FINGER, nID_L, nID_H, nFlag, null, null);
        }

        public byte FamIdentify(ref uint nID_L, ref uint nID_H)
        {
            byte nFlag = FamDefs.FLAG_1_N_MATCH;
            if (m_bMatchWithLog)
                nFlag = FamDefs.FLAG_1_N_MATCH_LOG;
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_MATCH_FINGER, 0, 0, nFlag, null, null);
            if (m_nErrorCode != 0)
                return m_nErrorCode;

            nID_L = (uint)( m_RxCmd[2] + (m_RxCmd[3] << 8) + (m_RxCmd[4] << 16) + (m_RxCmd[5] << 24) );
            nID_H = (uint)( m_RxCmd[6] + (m_RxCmd[7] << 8) + (m_RxCmd[8] << 16) + (m_RxCmd[9] << 24));
            return 0;
        }

        public byte FamGetUserListLength(ref uint nLength)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_USER_LIST, 0, 0, 1, null, null);
            if (m_nErrorCode == 0)
            {
                nLength = m_commSerial.DataBufferLength;
            }
            return m_nErrorCode;
        }

        public byte[] FamUserList
        {
            get
            {
                return m_commSerial.DataBuffer;
            }
        }

        public byte FamUploadToRam(uint nAddress, uint nLength, byte[] TxBuf)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_UPLOAD_TO_RAM, nAddress, nLength, 0, TxBuf, null);
        }

        public byte FamDownloadFromRam(uint nAddress, uint nLength, byte[] RxBuf)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_FROM_RAM, nAddress, nLength, 0, null, RxBuf);
        }

        public byte FamDownloadFromFlash(uint nAddress, uint nLength, byte[] RxBuf)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_FROM_FLASH, nAddress, nLength, 0, null, RxBuf);
        }

        public byte FamWriteToFlash(uint nLength)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_WRITE_TO_FLASH, nLength, 0, 0, null, null);
        }

        public byte FamReboot()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_SOFTWARE_REBOOT, 0, 0, 0, null, null);
        }

        public byte FamGetVersion(ref string szVerFw, ref string szVerHw)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_GET_VERSION, 0, 0, 0, null, null);
            if (m_nErrorCode == 0)
            {
                byte[] bySubVersion = new byte[1];
                bySubVersion[0] = m_RxCmd[7];
                szVerFw = string.Format("{0:d}.{1:d}", m_RxCmd[4], m_RxCmd[2] );
                szVerFw += System.Text.Encoding.Default.GetString(bySubVersion);
                szVerHw = string.Format("{0:d}.{1:d}", m_RxCmd[8], m_RxCmd[6]);
            }
            return m_nErrorCode;
        }

        public byte FamGetFS28Version(ref string szVerFw, ref string szVerHw)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_GET_FS28_VERSION, 0, 0, 0, null, null);
            if (m_nErrorCode == 0)
            {
                szVerFw = string.Format("{0:d}.{1:d}.{2:d}", m_RxCmd[4], m_RxCmd[3], m_RxCmd[2]);
                szVerHw = string.Format("{0:d}",  m_RxCmd[6]);
            }
            return m_nErrorCode;
        }

        public byte FamDeleteOneUser(uint nIDL, uint nIDH)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DELETE_1_USER, nIDL, nIDH, 1, null, null);
        }

        public byte FamDeleteAllUser()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_DELETE_ALL_USER, 0, 0, 0, null, null);
        }

        public byte FamChangeUserType(uint nIDL, uint nIDH, byte nUType)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_CHANGE_USER_TYPE, nIDL, nIDH, nUType, null, null);
        }

        public byte FamDownloadTemplateLength(uint nIDL, uint nIDH, ref uint nTemplateLength)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_DOWNLOAD_TEMPLATE, nIDL, nIDH, 0, null, null);
            if (m_nErrorCode == 0)
            {
                nTemplateLength = m_commSerial.DataBufferLength;
            }
            return m_nErrorCode;
        }

        public byte[] FamDownloadedTemplate
        {
            get
            {
                return m_commSerial.DataBuffer;
            }
        }

        public byte FamUploadTemplate(uint nTemplateLength, byte[] pTemplate)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_UPLOAD_TEMPLATE, 0, nTemplateLength, 0, pTemplate, null);
        }

        public byte FamGetSpace(ref uint nPages)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_GET_SPACE, 0, 0, 0, null, null);
            if (m_nErrorCode == 0)
                nPages = (uint)(m_RxCmd[2] + (m_RxCmd[3] << 8) + (m_RxCmd[4] << 16) + (m_RxCmd[5] << 24));
            return m_nErrorCode;
        }

        public byte FamGetSecurityLevel(ref byte nSLevel)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_SECURITY_LEVEL, 0, 0, 0, null, null);
            if (m_nErrorCode == 0)
                nSLevel = m_RxCmd[2];
            return m_nErrorCode;
        }

        public byte FamSetSecurityLevel(byte nSLevel)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_SECURITY_LEVEL, nSLevel, 0, 1, null, null);
        }

        public byte FamCheckLog(ref uint nLogNumber)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_ACCESS_LOG, 0, 0, FamDefs.FLAG_CHECK_LOG, null, null);
            if (m_nErrorCode == 0)
            {
                nLogNumber = (uint)( m_RxCmd[2] + m_RxCmd[3]*0x100 + m_RxCmd[4]*0x10000 + m_RxCmd[5]*0x1000000 );
            }
	        return m_nErrorCode;
        }

        public byte FamGetLog(uint nOffset, uint nNumber, byte[] pLog)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_ACCESS_LOG, nOffset, nNumber, FamDefs.FLAG_GET_LOG, null, pLog);	
        }

        public byte FamEraseLog()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_ACCESS_LOG, 0, 0, FamDefs.FLAG_ERASE_LOG, null, null);	
        }

        public byte FamSyncTime(int nYear, byte nMonth, byte nDay, byte nHour, byte nMin, byte nSec)
        {
	        uint nP1 = (uint) ( nSec + nMin*0x100 + nHour*0x10000 + nDay*0x1000000);
	        uint nP2 = (uint) ( nMonth + nYear*0x100 );
            return CommunicateWithFAC(FamDefs.COMMAND_GET_SET_TIME, nP1, nP2, FamDefs.FLAG_SET_ALL, null, null);	
        }
        
        public byte FS28RefreshTime()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_REFRESH_TIME, 0, 0, 0, null, null);	
        }

        public byte FamGetIR(ref byte nIR)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_NETWORK_SETTING, 0, 0, FamDefs.FLAG_GET_IR, null, null);
            if (m_nErrorCode == 0)
            {
                nIR = m_RxCmd[2];
            }
	        return m_nErrorCode;
        }
        public byte FS28ReadBattery(ref uint nBatterVol)
        {
            m_nErrorCode = CommunicateWithFAC(FamDefs.COMMAND_GET_BATTERY_VOLTAGE, 0, 0, FamDefs.FLAG_CHECK_LOG, null, null);
            if (m_nErrorCode == 0)
            {
                nBatterVol = (uint)(m_RxCmd[2] + m_RxCmd[3] * 0x100 + m_RxCmd[4] * 0x10000 + m_RxCmd[5] * 0x1000000);
            }
            return m_nErrorCode;
        }
        public byte FS28EnableSelfMsg(bool flag)
        {
            if(flag)
                return CommunicateWithFAC(FamDefs.COMMAND_ENABLE_SELF_DEFINE_MSG, 0, 0, 1, null, null);
            else
                return CommunicateWithFAC(FamDefs.COMMAND_ENABLE_SELF_DEFINE_MSG, 0, 0, 0, null, null);
        }
        public byte FS28SendMsg(uint nMsgLength, byte Postion, byte[] pMsg)
        {
            return CommunicateWithFAC(FamDefs.COMMAND_SEND_MSG, 0, nMsgLength, Postion, pMsg, null);
        }
        public byte FS28ClearMsg()
        {
            return CommunicateWithFAC(FamDefs.COMMAND_CLEAR_MSG, 0, 0, 0, null, null);
        }
        public string ErrorMessage
        {
            get
            {
                if (m_commSerial.IsComError)
                {
                    m_strErrorMessage = m_commSerial.ComErrorMessage;
                    return m_strErrorMessage;
                }

                switch (m_nErrorCode)//error code from flag of FAM Respond
                {
                    case FamDefs.RET_NO_IMAGE:
                        m_strErrorMessage = "Respond Fail or Not Image!";
                        break;
                    case FamDefs.RET_BAD_QUALITY:
                        m_strErrorMessage = "Bad Quality!";
                        break;
                    case FamDefs.RET_TOO_LITTLE_POINTS:
                        m_strErrorMessage = "Too littlt points!";
                        break;
                    case FamDefs.RET_EMPTY_BASE:
                        m_strErrorMessage = "Empty database!";
                        break;
                    case FamDefs.RET_UNKNOWN_USER:
                        m_strErrorMessage = "Unknown user!";
                        break;
                    case FamDefs.RET_NO_SPACE:
                        m_strErrorMessage = "Not enough memory!";
                        break;
                    case FamDefs.RET_BAD_ARGUMENT:
                        m_strErrorMessage = "Bad argument!";
                        break;
                    case FamDefs.RET_CRC_ERROR:
                        m_strErrorMessage = "CRC error!";
                        break;
                    case FamDefs.RET_RXD_TIMEOUT:
                        m_strErrorMessage = "Rx data time out!";
                        break;
                    case FamDefs.RET_USER_ID_IS_ABSENT:
                        m_strErrorMessage = "User id does NOT existed!";
                        break;
                    case FamDefs.RET_USER_ID_IS_USED:
                        m_strErrorMessage = "User id existed!";
                        break;
                    case FamDefs.RET_VERY_SIMILAR_SAMPLE:
                        m_strErrorMessage = "Sample is very similar!";
                        break;
                    case FamDefs.RET_USER_SUSPENDED:
                        m_strErrorMessage = "User is suspended!";
                        break;
                    case FamDefs.RET_UNKNOWN_COMMAND:
                        m_strErrorMessage = "Unknown command!";
                        break;
                    case FamDefs.RET_INVALID_STOP_BYTE:
                        m_strErrorMessage = "Invalid stop byte!";
                        break;
                    case FamDefs.RET_HARDWARE_ERROR:
                        m_strErrorMessage = "Hardware error!";
                        break;
                    case FamDefs.RET_BAD_FLASH:
                        m_strErrorMessage = "Bad flash!";
                        break;
                    case FamDefs.RET_TOO_MANY_VIP:
                        m_strErrorMessage = "Too many VIP!";
                        break;
                    case FamDefs.RET_CONNECT_TIMEOUT:
                        m_strErrorMessage = "Time out to connect to FAM!";
                        break;
                    case FamDefs.RET_FLAG_FS28_BUSY:
                        m_strErrorMessage = "FS28 is Busy! Pls Try again later";
                        break;
                    case FamDefs.RET_FLAG_FS28_SLEEP:
                        m_strErrorMessage = "FS28 just weak up from sleep!Pls Try 5 sec later";
                        break;
                    case FamDefs.RET_FLAG_FS28_IN_SECURITY_LOCKED:
                        m_strErrorMessage = "FS28 is security locked!Pls unlock it";
                        break;                       
                    case FamDefs.RET_FLAG_9866B_OFF:
                        m_strErrorMessage = "9866B off. Pls reset the COM";
                        break;
                    default:
                        m_strErrorMessage = string.Format("Unknown error code 0x: {0:x}", m_nErrorCode);
                        break;
                }
                return m_strErrorMessage;
            }
        }
    }
}
